<?php
/**
 * Contact Form Submission API Endpoint
 * Handles contact form submissions and stores them in the database.
 * * NOTE: For production, store credentials outside the web root (e.g., in environment variables).
 * This file is self-contained for easy testing.
 */

// --- 1. Database Configuration ---
// Update these values with your actual database credentials
define('DB_HOST', 'localhost');
define('DB_NAME', 'deepgpu_liv');
define('DB_USER', 'deepgpu_liv');
define('DB_PASS', 'Tower@32');
define('DB_CHARSET', 'utf8mb4');

// --- 2. Headers and Preflight Handling ---
header('Access-Control-Allow-Origin: *'); // Allow requests from any domain
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Content-Type: application/json');

// Handle preflight requests (required for CORS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// --- 3. Helper Functions ---

/**
 * Creates and returns a database connection (PDO object).
 * Exits the script with a JSON error response on failure.
 * @return PDO The active database connection.
 */
function getConnection() {
    try {
        $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false, // Crucial for security
        ];
        
        $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        return $pdo;
    } catch (PDOException $e) {
        // Log the detailed error message privately
        error_log("Database connection failed: " . $e->getMessage()); 
        
        // Send a generic error response to the client
        http_response_code(500);
        sendResponse(false, 'Database connection failed');
    }
}

/**
 * Validates an email address format.
 * @param string $email The email string to validate.
 * @return bool True if valid, false otherwise.
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

/**
 * Sanitize input data against XSS and prepares for storage.
 * @param string $data The input string.
 * @return string The sanitized string.
 */
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    // Convert special characters to HTML entities
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8'); 
    return $data;
}

/**
 * Get client IP address, checking various server headers.
 * @return string The client IP address or 'UNKNOWN'.
 */
function getClientIP() {
    $ipaddress = '';
    if (isset($_SERVER['HTTP_CLIENT_IP']))
        $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
    else if(isset($_SERVER['HTTP_X_FORWARDED_FOR']))
        $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
    else if(isset($_SERVER['REMOTE_ADDR']))
        $ipaddress = $_SERVER['REMOTE_ADDR'];
    else
        $ipaddress = 'UNKNOWN';
    return $ipaddress;
}

/**
 * Sends a JSON response to the client and terminates the script.
 * @param bool $success Operation status.
 * @param string $message Status message.
 * @param array|null $data Optional data payload.
 */
function sendResponse($success, $message, $data = null) {
    $response = [
        'success' => $success,
        'message' => $message
    ];
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit();
}

// --- 4. Main API Logic ---

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    sendResponse(false, 'Method not allowed. Only POST is accepted.');
}

// Get JSON input from the request body
$input = json_decode(file_get_contents('php://input'), true);

// Check if JSON decoding failed or input is empty
if ($input === null) {
    http_response_code(400);
    sendResponse(false, 'Invalid JSON input or empty body.');
}

// Validate required fields (checking for existence and non-empty string)
if (empty($input['fullName']) || empty($input['email']) || empty($input['subject']) || empty($input['message'])) {
    http_response_code(400);
    sendResponse(false, 'Missing required fields: fullName, email, subject, and message are mandatory.');
}

// Sanitize inputs
$fullName = sanitizeInput($input['fullName']);
$email = sanitizeInput($input['email']);
$subject = sanitizeInput($input['subject']);
$message = sanitizeInput($input['message']);

// Handle optional fields, setting to null if not provided
$phone = isset($input['phone']) ? sanitizeInput($input['phone']) : null;
$company = isset($input['company']) ? sanitizeInput($input['company']) : null;

// Validate email format
if (!validateEmail($email)) {
    http_response_code(400);
    sendResponse(false, 'Invalid email address format.');
}

// Get client information
$ipAddress = getClientIP();
$userAgent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : null;

try {
    // Attempt to establish database connection
    $pdo = getConnection();
    
    // SQL query using named placeholders for security
    $sql = "INSERT INTO contact_submissions 
            (full_name, email, phone, company, subject, message, ip_address, user_agent) 
            VALUES 
            (:full_name, :email, :phone, :company, :subject, :message, :ip_address, :user_agent)";
    
    $stmt = $pdo->prepare($sql);
    
    // Execute the prepared statement with sanitized data
    $stmt->execute([
        ':full_name' => $fullName,
        ':email' => $email,
        ':phone' => $phone,
        ':company' => $company,
        ':subject' => $subject,
        ':message' => $message,
        ':ip_address' => $ipAddress,
        ':user_agent' => $userAgent
    ]);
    
    $submissionId = $pdo->lastInsertId();
    
    // Success response
    http_response_code(201); // 201 Created
    sendResponse(true, 'Contact form submitted successfully', [
        'id' => $submissionId
    ]);
    
} catch (PDOException $e) {
    // Log the specific database error and send a generic failure message
    error_log("Contact form PDO error: " . $e->getMessage());
    http_response_code(500);
    sendResponse(false, 'Failed to submit contact form due to a server error.');
}
?>